function [bestw bestppi bestigmax]=ppoptimize4(x,y,varargin)
% PPOPTIMIZE runs optimalization proces of PPI function for given dataset X and labels Y.
%
%   W = PPOPTIMIZE4(X,Y,PARAMETERS,VALUE,...)
%	searching for optimal W witch maximize PP index value
%	X : dataset
%	Y : labels
%
%	posible PARAMETERS are:
%	   Function:
%		'function' - 'triangle','fx4' or 'bicentral' are recognized
%		           as f_triangular, f_x4 and f_bicentral function accordinly (default f_x4)
%		'beta'     - value of BETA parameter - width of given 'function'
%		           (default 2, seams to be good value for normalized data and f_x4 function)
%
%     Optimalization:
%		'maxIterations'	  - maximum number of iterations (default 1000)
%       'initiations'     - number of repetitions of optimalization, best
%                       solusion over those initiations is choosen at the
%                       end of training
%		'learningRate'    - learning rate for gradnient descent procedure (default 0.1)
%		'initWeights'     - initial weights (staring point)
%       'stopCriterium'   - choose criterium for ending of optimalization (default 2), 
%                         currently implemented are:
%                         1 - check difference beetwen two last values,
%                         if |PPI(t)-PPI(t-1)| < eps then stop
%                         2 (default) check difference beetwen average
%                         values over 'checkPeriod' last values
%                         
%       'checkPeriod'     - if 'stopCriterium' = 2 then optimalization
%                         stops when average value of ppifunction over checkPeriod last
%                         values is considered. If difference beetwen two last average values
%                         is less (default: 5)
%       'eps'             - set eps value (default 0.001), for given stoping
%		                  criterium define max. difference between values of (or average values over checkPeriod) PPIFUNCTION
%       
%     Searching for ortogonal solusion:
%        'ortoWeights'    - weights for witch ortgonal solution will be searched  (default [])
%        'lambda'         - value of LAMBDA parameter controls an influence of ortogonalization process 
%
%     Other optimalization parameters;
%        'iGmax'          - index of input vector for with maximum value of G
%                         function will be searched, required in secound
%                         stage of learning (under development)
%
%     Plotting:
%		 'plot'           - currently implemented plots: (default 'none') 
%                        'all' - plots all on single figure for each iteration, 
%                        'ppi' - plots PPIFUNCTION for each iteration,
%                        'last' - plots final projection 
%      
%     Results saving:
%        'log'            - 'off' or 'on' (default 'off')
%        'savedir'        - name of directory to store logfiles and
%                         pictures created during learning process.
%                         Default dir name is: 'yy.mm.dd-ppi-results',
%                         where yy.mm.dd is the current date.
%        'save'        -  'all', 'last' or 'none' - saves  pictures in SAVEDIR with
%                          apropriate names (default 'none'). Note thet if SAVEALL = YES then 
%                         option PLOT is automatic set to ALL.
%        'display'        - 'none', 'short', 'all' degree of informations  displayed on screan  (default 'short') 
%        'logFile'        - name of file with stored results (default: data.log if no dataname is given)
%        'dataName'       - name of dataset, used for names in output files
%                         (logs and pictures), (default: 'data')
%
%	e.g.   w=ppoptimize(x,y,'function','triangular','beta',3,'plot','ppi')
%		use triangular function with beta = 3 and plot change of ppi function douring learning (function width)
%

% TODO : 
%   * document all parameters and options (almost done)
%   * implementation of diferent optimalization methods: e.g. sim. aneling
%   (see MATLAB docs)
%   * simple gui for controling and visualization of optimalization
%   * new pictures - G values, weights values as bar plot, ... 
%   * improve log saving and results displaing   
%
param = inputParser;
% data
param.addRequired('x',@isnumeric);
param.addRequired('y',@isnumeric);

% inner function
param.addOptional('function','f_x4',@(x)any(strcmpi(x,{'triangle','f_x4','fx4','bicentral'})));
param.addOptional('beta',10, @(x)isnumeric(x) && x>0);

% optimalization global
param.addParamValue('learningRate',0.1,@(x)isnumeric(x) && x >= 0);
param.addParamValue('eps',0.001,@(x)isnumeric(x) && x>0);
param.addParamValue('maxIterations',1000,@(x)isnumeric(x) && x>0 && mod(x,1)==0);
param.addParamValue('initiations',5,@(x)isnumeric(x) && x > 0 && mod(x,1)==0);
param.addParamValue('checkPeriod',5,@(x)isnumeric(x) && x>0 && mod(x,1)==0);
param.addParamValue('stopCriterium',2,@(x)x==1|| x==2);

% optimalization - case depend
param.addParamValue('initWeights', [],@(x)isnumeric(x));
param.addParamValue('lambda', 0.1,@(x)isnumeric(x) && x > 0 );
param.addParamValue('ortoWeights',[],@isnumeric);
param.addParamValue('indGmax',[],@isnumeric);

% loging
param.addParamValue('log','off',@(x)any(strcmpi(x,{'on','off'})));
param.addParamValue('logFile',[],@ischar);
param.addParamValue('dataName','data',@ischar);
param.addParamValue('save','none',@(x)any(strcmpi(x,{'none','all','last'})));
param.addParamValue('savedir',[],@ischar);
param.addParamValue('display','short',@(x)any(strcmpi(x,{'none','all','short'})));

% plotting
param.addParamValue('plot','none',@(x)any(strcmpi(x,{'none','all','ppi','last'})));
%
param.parse(x,y,varargin{:});
%fprintf('Input parameters:\n\n');
%disp(param.Results);
%disp(param.Parameters);

[vx fx]=size(x);

beta        = param.Results.beta;           
lrate       = param.Results.learningRate;	% learning rate (step of gradnient descent)
eps         = param.Results.eps;	% 
nmax        = param.Results.maxIterations;     % nax. number of iterations
%pplot       = 0;
%lastppi     = -1;
plotall     = 0;
plotlast    = 0;
iplot       = 0;
% ffplot      = 0;
ninit       = param.Results.initiations;
ww          = param.Results.initWeights;
dataname    = param.Results.dataName;
avgtest     = param.Results.checkPeriod;
stopcriterium = param.Results.stopCriterium;
iGmax       = param.Results.indGmax;
wort        = param.Results.ortoWeights;
lambda      = param.Results.lambda;
orto        = ~isempty(wort);
funcname    = param.Results.function;
%wo          = 0;
saveall     = 0;
savelast    = 0;
savedir     = strcat(datestr(now,'yy.mm.dd'),'-ppi-results');
%procedure   = '';
log         = 0;

bestw = [];
bestppi = [];
bestinit = 0;
bestn = 0;
bestigmax = -1;

switch param.Results.display
    case 'all'
        display = 2;
    case 'short'
        display = 1;
    case 'none'
        display = 0;
end

switch param.Results.log
    case 'on'
        log = 1;
    case 'off'
        log = 0;
end

switch funcname
    case { 'fx4' , 'f_x4'}
        func = @(xx)f_x4(xx,beta);
    case 'triangle'
        func = @(xx)f_triangular(xx,beta);
    case 'bicentral'
        func = @(xx)f_bicentral(xx,beta,0,10);
    otherwise
        error('Ten blad nie powinien wystapic - ale jesli wystapil to znaczy, ze jest cos nie tak z podana funkcja');
end

switch param.Results.plot
    case 'all'
        plotall = 1;
    case 'ppi'
        iplot = 1;
    case 'last'
        plotlast = 1;
end

if ~isempty(param.Results.savedir)
    savedir = param.Results.savedir;
end

switch param.Results.save
    case 'all'
        plotall = 1;
        saveall = 1;
        log = 1;
    case 'last'
        plotlast = 1;
        savelast = 1;
        log = 1;
    case 'iplot'
        iplot = 1;
end


if isempty(iGmax) 
    if orto == 0
        procedure = 'ppi';
        ppifun = @(wx)ppifunction(x,y,wx,func);    % podstawowa postac indeksu
    else
        procedure = strcat('ppi-orto',sprintf('.%0.1f',lambda));   % indeks 
        wort=wort/norm(wort);
        ortoproj = x*wort';
        ppifun = @(wx)ppifunction(x,y,wx,func,wort,lambda);
    end
else 
    procedure = strcat('ppi-gmax',sprintf('.%d',iGmax));
    ppifun = @(wx)ppigmaxfunction2(x,y,wx,iGmax,func);
end


text = '';
if log == 1 || display > 0
    text = [sprintf('procedure     = %s\n',procedure)...     
            sprintf('dataname      = %s\n',dataname)...
            sprintf('vectors       = %d\n',vx)...        
            sprintf('features      = %d\n',fx)...            
            sprintf('learningRate  = %f\n',lrate)...
            sprintf('eps           = %f\n',eps)...           
            sprintf('initiations   = %d\n',ninit)...
            sprintf('maxIterations = %d\n',nmax)...          
            sprintf('ppi function  = %s\n',func2str(ppifun))...
            sprintf('function      = %s\n',func2str(func))...
            sprintf('beta          = %f\n',beta)...          
        ];
    if orto == 1
        text = [text 'ortogonal W   = ' sprintf('%f  ',wort) '\n'...
                sprintf('lambda        = %f\n',lambda)...
        ]; 
    end
    if ~isempty(iGmax)
        text = [text sprintf('Gmax index    = %d\n',iGmax)];
    end
end

if log == 1
    [s comment]=mkdir(savedir);
    if s == 0
        error(comment);
    end
    prefix = strcat(savedir,'/',dataname,sprintf('.%s-%s.%0.1f',procedure,funcname,beta));
    logfilename = param.Results.logFile;
    if isempty(logfilename)
        logfilename = strcat(prefix,'.log');
    end
    logfile = fopen(logfilename,'wt');
    if logfile == -1
        error('Error opening %s file\n',logfilename);
    end
    fprintf(logfile,text);
end

if display > 0
    fprintf(text);
end

%%%%%%%%%%%%%%   OK - tu zaczyna sie zabawa %%

for initcount=1:ninit
	n = 0;
	ppitable = zeros(1,nmax+1);

    if display > 0 || log == 1
        text = sprintf('\nInitialization %d of %d\n',initcount,ninit);
        if display > 0
            fprintf(text);
        end
        if log == 1
            fprintf(logfile,text);
        end
    end
    
    if isempty(ww)
        w = rand(1,fx)*2-1;       % random initialization [-1,1]
    else
		w = ww;
    end

    avgppi = 0;
    lastavgppi = 0;
        
    while ( 1 )
        if (mod(n,avgtest) == 0 && n > 0)
            lastavgppi = avgppi;
            avgppi = mean(ppitable(n-avgtest+1:n));
        end;

        w=w/norm(w);
        [ppi ppid projection G] = ppifun(w);
		
        if nargout > 2
            [gmax igmax] = max(G);
        end
        
        lastppi = ppi;
		ppitable(1,n+1) = ppi;
	
        if iplot == 1 && n > 0
			plotppi(ppitable,n,avgtest);
            drawnow;
        end

        if orto == 1
            w1 = w*wort';
        end

        if display > 1 || (display == 1 && n == 0)
            fprintf('%3d  %10.6f   ',n,ppi);   fprintf('  %6.4f',w);
            if orto == 1
                fprintf('  [ %6.4f ] ',w1);
            end
            fprintf('\n');
        end
        if log == 1
            fprintf(logfile,'%3d  %10.6f   ',n,ppi);   fprintf(logfile,'  %6.4f',w);
            if orto == 1
                fprintf(logfile,'  [ %6.4f ] ',w1);
            end
            fprintf(logfile,'\n');
        end
        
        if plotall == 1
            str=cell(1,3);
            clf;
            set(gcf,'Color','w');
            if orto == 0
                
                str{1} = strcat('w = ',sprintf(' %.2f ',w));
                str{2} = strcat('I = ',sprintf(' %.5f ',ppitable(n+1)));
                str{3} = strcat('N = ',sprintf(' %d ',n));
 
                bgraph3(projection,y,'position',[0.05 0.4 0.9 0.50],'sigma',0.01,'function',func,'str',str);
            else
                str{2} = strcat('\alpha = ',sprintf(' %.4f ',w1));
                str{1} = strcat('I = ',sprintf(' %.4f ',ppitable(n+1)));
                str{3} = strcat('N = ',sprintf(' %d ',n));
                
                
                scaterplot([projection ortoproj],y,'str',str);
                
                set(gca,'position',[0.1 0.45 0.85 0.50]);
                xlabel(strcat('w2 = ',sprintf(' %.2f ',w)));
                ylabel(strcat('w1 = ',sprintf(' %.2f ',wort)));
            end
            axes('position',[0.1 0.1 0.85 0.25]);
            plotppi(ppitable,n,50);
            drawnow();
            
            if saveall == 1
                saveplot(strcat(prefix,sprintf('-%d.frame%04d',initcount,n)));
            end
        end;
        
        if (stopcriterium == 1 && (abs((ppi - lastppi)/lastppi) < eps) && n > 1 ) 
            break;
        end;
        if (stopcriterium == 2 && abs((avgppi - lastavgppi)/lastavgppi) < eps && n > 1 ) 
            break;
        end;
        if ( n >= nmax ) 
            break;
        end
        
        w = w + lrate * ppid;
        n = n + 1;
        
    end;
	
    if (display == 1)  % if display == 1 then display only last result
        fprintf('%3d  %10.6f   ',n,ppi); 
        fprintf('  %6.4f',w); 
        if orto == 1
                fprintf('  [ %6.4f ] ',w1);
        end
        fprintf('\n');
    end
    
	if plotlast == 1
            str=cell(1,3);
            clf;
            set(gcf,'Color','w');
            if orto == 0
                
                str{1} = strcat('w = ',sprintf(' %.2f ',w));  % !! dla duzej ilosci wag wychodzi sieczka
                str{2} = strcat('I = ',sprintf(' %.5f ',ppitable(n+1)));
                str{3} = strcat('N = ',sprintf(' %d ',n));
 
                bgraph3(projection,y,'position',[0.05 0.4 0.9 0.50],'sigma',0.01,'function',func,'str',str);
            else
                str{2} = strcat('\alpha = ',sprintf(' %.4f ',w1));
                str{1} = strcat('I = ',sprintf(' %.4f ',ppitable(n+1)));
                str{3} = strcat('N = ',sprintf(' %d ',n));
                
                
                scaterplot([projection ortoproj],y,'str',str);
                
                set(gca,'position',[0.1 0.45 0.85 0.50]);
                xlabel(strcat('w2 = ',sprintf(' %.2f ',w)));
                ylabel(strcat('w1 = ',sprintf(' %.2f ',wort)));
                
            end
            axes('position',[0.1 0.1 0.85 0.25]);
            plotppi(ppitable,n,50);
            drawnow();
            
            if savelast == 1
                saveplot(strcat(prefix,sprintf('-%d.last',initcount)));
            end
	end;
	
	if (initcount == 1) || (ppi > bestppi) 
		bestw = w;
		bestppi = ppi;
		bestinit = initcount;
		bestn = n;
        if nargout > 2
            bestigmax = igmax;
        end
	end
end

if display > 0 || log == 1
    text = ['---------------------------------------------\n' ...
            sprintf('\nInitialization  %d was the best\n',bestinit)...
            sprintf('%d  %6.4f    ',bestn,bestppi)...
            sprintf('  %6.4f',bestw) sprintf('\n')];
    if display > 0
        fprintf(text);
    end
    if log == 1
        fprintf(logfile,text);
        fclose(logfile);
    end
end


function saveplot(prefix)
 	name = strcat(prefix,'.png');
 	print('-dpng','-r96',name);
% % octave only
% %	print('-dpng','-S640,480',name);


%    name = strcat(prefix,'.eps');
%  	 print('-depsc',name);

function plotppi(ppitable,n,scalefactor)
    if nargin < 3
        scalefactor = 10;
    end
    if nargin < 2
        n = length(ppitable);
    end
    xl = scalefactor.*(fix(n./scalefactor)+1);
    cla;
    hold on;
    xlim([0 xl]);
    xlabel('Iterations');
    ylabel('PPI value');
    plot(0:n,ppitable(1:n+1),'-b','LineWidth',2);
    box on;
 	hold off;
