function [parameters unmatched]=qpc_config(varargin)
%   Returns configuration struct for QPC.
%	Posible PARAMETERS are:
%	   Function:
%		'function' - 'gauss'(default) , 'triangle','fx4' or 'bicentral' 
%		'beta'     - value of BETA parameter - width of given 'function'
%		           (default .1, seams to be good value for normalized data
%		           and gauss function. For fx4 try beta=2 and for bicentral
%		           beta=?)
%
%     Optimalization:
%       'OptMethod'       - optimization method.
%                           'gd' - gradnient descent (default, build-in method)
%                           'newton' or 'fminunc'- Quasi-Newton method
%                                ('fminunc' with 'GradObj'='on') 
%                            'random' - random hill climbing method.
%                           'fminsearch' use fminsearch optimization
%
%       'OptConf'           - optimization configuration given by 'optiset'
%                            (default [])
%       'QPCMethod'         - QPC method:
%                            'qpc'  - standard QPC function O(d n^2). 
%                            'quick' - QPC estimation for intervals O(knd)
%                             Note that only 'fminsearch' and 'random'
%                             optimization methods are suported for QPCquick.
%                            'proto'  - QPC with prototypes O(knd).
%                     
%		'maxIterations'	  - maximum number of iterations (default 1000)
%       'initiations'     - number of repetitions of optimalization, best
%                       solusion over those initiations is choosen at the
%                       end of training of each projection (default 5)
%		'learningRate'    - weights learning rate for ('gd') gradnient descent procedure (default 0.1)
%		'initWeights'     - initial weights (default [], random initialization)
%       'prototypes'      - [KxN] matrix with prototypes positions (K -
%                           number of prototyprs, N-1 features, last column
%                           contines class labels associatet with prototypes)
%       'plr              - prototypes learning rate for ('gd') gradient
%                          descent procedure (default 0, prorotypes don't
%                          change thier positions).
%                          Note: currently only 'gd' is supported for plr > 0.
%       'stopCriterium'   - choose criterium for ending of optimalization (default 2), 
%                         currently implemented are:
%                         1 - check difference beetwen two last values,
%                         if |QPC(t)-QPC(t-1)| < eps then stop
%                         2 (default) check difference beetwen average
%                         values over 'checkPeriod' last values
%                         
%       'checkPeriod'     - if 'stopCriterium' = 2 then optimalization
%                         stops when average value of qpcfunction over checkPeriod last
%                         values is considered. If difference beetwen two last average values
%                         is less (default: 5)
%       'eps'             - set eps value (default 0.001), for given stoping
%		                  criterium define max. difference between values of (or average values over checkPeriod) QPCFUNCTION
%       
%     Searching for ortogonal solusion:
%        'ortoWeights'    - weights for witch ortgonal solution will be searched  (default [])
%        'lambda'         - value of LAMBDA parameter controls an influence of ortogonalization process 
%
%     Other optimalization parameters;
%        'iGmax'          - index of input vector for with maximum value of G
%                         function will be searched, required in secound
%                         stage of learning (under development)
%
%     Plotting:
%		 'plot'           - currently implemented plots: (default 'none') 
%                        'all' - plots all on single figure for each iteration, 
%                        'qpc' - plots QPCFUNCTION for each iteration,
%                        'last' - plot only final projection 
%      
%     Program verbse level:
%        'display'        - 'none', 'short', 'all' degree of informations  displayed on screan  (default 'none') 
%
%     Results log:
%        'log'            - 'off' or 'on', log to file (default 'off')
%        'savedir'        - name of directory to store logfiles and
%                         pictures created during learning process.
%                         Default dir name is: 'yy.mm.dd-qpc-results',
%                         where yy.mm.dd is the current date.
%        'save'        -  'all', 'last' or 'none' - saves  pictures in SAVEDIR with
%                          apropriate names (default 'none'). Note thet if SAVEALL = YES then 
%                         option PLOT is automatic set to ALL.
%        'logFile'        - name of file with stored results (default: data.log if no dataname is given)
%        'dataName'       - name of dataset, used for names in output files
%                         (logs and pictures), (default: 'data')
% TODO:
%   * czy parametr indGmax jest uzywany przy uczeniu QPC (lub innych
%   funkcji) - moze da sie zastapic prototypem [x(indGmax,:) y(indGmax))],
%   przez co staje sie bardziej ogolny.


    param = inputParser;
    param.KeepUnmatched = true;
    % data
%    param.addRequired('x',@isnumeric);
%    param.addRequired('y',@isnumeric);

    % inner function
    param.addOptional('function','gauss',@(x)any(strcmpi(x,{'triangle','f_x4','fx4','bicentral','gauss'})));
    param.addOptional('beta',0.1, @(x)isnumeric(x) && x>0);

    % optimalization global
    param.addParamValue('OptMethod','gd',@(x)any(strcmpi(x,{'newton','gd','fminunc','fminsearch','random'})));
    param.addParamValue('QPCMethod','qpc',@(x)any(strcmpi(x,{'qpc','quick','proto','uqpc1','uqpc2'})));
    param.addParamValue('OptConf',[],@(x)isstruct(x));
    param.addParamValue('learningRate',0.1,@(x)isnumeric(x) && x >= 0);
    param.addParamValue('eps',0.001,@(x)isnumeric(x) && x>0);
    param.addParamValue('maxIterations',1000,@(x)isnumeric(x) && x>0 && mod(x,1)==0);
    param.addParamValue('multistart','yes',@(x)any(strcmpi(x,{'yes','no'})));
    param.addParamValue('killPeriod',10,@(x)isnumeric(x) && x>0 && mod(x,1)==0);
    param.addParamValue('killRatio',0.5,@(x)isnumeric(x) && x>0 && mod(x,1)==0);
    param.addParamValue('initiations',10,@(x)isnumeric(x) && x > 0 && mod(x,1)==0);
    param.addParamValue('directions',2,@(x)isnumeric(x) && x>0 && mod(x,1)==0);
    param.addParamValue('checkPeriod',5,@(x)isnumeric(x) && x>0 && mod(x,1)==0);
    param.addParamValue('stopCriterium',2,@(x)x==1|| x==2);

    % optimalization - case depend
    param.addParamValue('initWeights', [],@(x)isnumeric(x));
    param.addParamValue('prototypes', [],@(x)isnumeric(x));
    param.addParamValue('plr',0,@(x)isnumeric(x));
    param.addParamValue('lambda', 0.1,@(x)isnumeric(x) && x > 0 );
    param.addParamValue('ortoWeights',[],@isnumeric);
    param.addParamValue('indGmax',[],@isnumeric);
    param.addParamValue('orthogonalizationMethod','projection',@(x)any(strcmpi(x,{'projection','error'})));

    % loging
    param.addParamValue('log','off',@(x)any(strcmpi(x,{'on','off'})));
    param.addParamValue('logFileName',[],@ischar);
    param.addParamValue('dataName','data',@ischar);
    param.addParamValue('save','none',@(x)any(strcmpi(x,{'none','all','last'})));
    param.addParamValue('savedir',[],@ischar);
    param.addParamValue('display','none',@(x)any(strcmpi(x,{'none','all','short'})));

    % plotting
    param.addParamValue('plot','none',@(x)any(strcmpi(x,{'none','all','qpc','last'})));
    %

param.parse(varargin{:});
%disp(param.Results);
parameters = param.Results;
unmatched = param.Unmatched;
unames=fieldnames(unmatched);
clear param;

if strcmp(parameters.QPCMethod,'qpc') && (~isempty(parameters.prototypes) || parameters.plr > 0)
    parameters.QPCMethod = 'proto';
end

if nargout  < 2 && size(unames,1) > 0
        error(['Unknown options: ' sprintf(' "%s"',unames{:})]);
end
